<?php
// setup_restructuring.php
require_once __DIR__ . '/includes/db.php';

echo "<h2>Iniciando Reestructuración de Base de Datos...</h2><pre>";

try {
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // 1. Verificamos y actualizamos la tabla 'vehiculos'
    echo "1. Verificando tabla 'vehiculos'...\n";

    // Crear si no existe
    $sqlCreate = "
    CREATE TABLE IF NOT EXISTS `vehiculos` (
      `id` int(11) NOT NULL AUTO_INCREMENT,
      `cliente_id` int(11) NOT NULL,
      `patente` varchar(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `marca_modelo` varchar(150) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `anio` int(4) DEFAULT NULL,
      `ultimo_kms` varchar(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
      `fecha_actualizacion` datetime DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      UNIQUE KEY `patente` (`patente`),
      KEY `cliente_id` (`cliente_id`)
    ) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;
    ";
    $pdo->exec($sqlCreate);
    echo "Tabla 'vehiculos' asegurada.\n";

    // Verificar columnas (por si tabla existía pero faltaba 'anio')
    $cols = $pdo->query("DESCRIBE vehiculos")->fetchAll(PDO::FETCH_COLUMN);
    if (!in_array('anio', $cols)) {
        $pdo->exec("ALTER TABLE vehiculos ADD COLUMN anio INT(4) DEFAULT NULL AFTER marca_modelo");
        echo "Columna 'anio' agregada.\n";
    }

    // 2. Migración de Datos desde Órdenes (Solo si está vacía o para actualizar nuevos)
    echo "2. Migrando vehículos desde órdenes de trabajo...\n";

    // Esta consulta extrae patentes únicas de todas las órdenes y las inserta/actualiza en vehiculos
    // COALESCE(NULLIF(patente, ''), NULLIF(vehiculo, '')) asegura que tengamos algún identificador
    $sqlMigrate = "
        INSERT INTO vehiculos (cliente_id, patente, marca_modelo, ultimo_kms, fecha_actualizacion)
        SELECT 
            cliente_id, 
            UPPER(TRIM(patente)), 
            TRIM(vehiculo), 
            kms, 
            MAX(fecha)
        FROM (
            SELECT cliente_id, patente, vehiculo, kms, fecha FROM ordenes_frenos WHERE patente != ''
            UNION ALL
            SELECT cliente_id, patente, vehiculo, kms, fecha FROM ordenes_motor WHERE patente != ''
            UNION ALL
            SELECT cliente_id, patente, vehiculo, kms, fecha FROM ordenes_suspension WHERE patente != ''
            UNION ALL
            SELECT cliente_id, patente, vehiculo, kms, fecha FROM ordenes_otros WHERE patente != ''
        ) as all_orders
        GROUP BY UPPER(TRIM(patente))
        ON DUPLICATE KEY UPDATE
            marca_modelo = VALUES(marca_modelo),
            ultimo_kms = VALUES(ultimo_kms),
            fecha_actualizacion = VALUES(fecha_actualizacion)
            -- No actualizamos cliente_id para no mover vehículos si el dueño cambió manualmente en esta tabla
    ";
    $stmt = $pdo->prepare($sqlMigrate);
    $stmt->execute();
    echo "Migración completada. Filas afectadas: " . $stmt->rowCount() . "\n";

    // 3. Crear Vista Lógica de Historial (Reparaciones)
    // MySQL View para facilitar consultas de 'Hoja de Vida'
    echo "3. Creando Vista 'view_historial_global'...\n";

    // Eliminamos vista si existe
    $pdo->exec("DROP VIEW IF EXISTS view_historial_global");

    // NOTA: Mysql 5.x/8.x soporta vistas.
    // Unimos todas las tablas. Agregamos columna 'tipo' para saber origen.
    $sqlView = "
    CREATE VIEW view_historial_global AS
        SELECT id as orden_id, 'Frenos' as tipo, cliente_id, fecha, vehiculo, patente, kms, total, orden_num, detalles_items as detalles FROM ordenes_frenos
        UNION ALL
        SELECT id as orden_id, 'Motor' as tipo, cliente_id, fecha, vehiculo, patente, kms, total, orden_num, detalles_items as detalles FROM ordenes_motor
        UNION ALL
        SELECT id as orden_id, 'Suspensión' as tipo, cliente_id, fecha, vehiculo, patente, kms, total, orden_num, detalles_items as detalles FROM ordenes_suspension
        UNION ALL
        SELECT id as orden_id, 'Otros' as tipo, cliente_id, fecha, vehiculo, patente, kms, total, orden_num, NULL as detalles FROM ordenes_otros
    ";
    $pdo->exec($sqlView);
    echo "Vista 'view_historial_global' creada exitosamente.\n";

    echo "<h3>Reestructuración Completada Correctamente.</h3>";

} catch (PDOException $e) {
    echo "Error Crítico: " . $e->getMessage();
}
echo "</pre>";
?>